#!/bin/ksh

# diddle.ksh
# version 1.1.12 2010-03-12

# System configuration and performance analysis for share Web hosting environments.
# Copyright (C) 2010, KrazyWorks, LLC
# igor@krazyworks.com
# 
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/


conf() {
	cache=600
}

perf() {
	sysname=$(hostname)

	# --------------------------------------
	# Attempt to identify Linux flavor
	# --------------------------------------

	osflavor="Some Linux"
	for i in /proc/version /var/log/dmesg /var/log/boot.msg
	do
		if [ -r "${i}" ]
		then
			if [ `grep -c "Linux version" "${i}"` -gt 0 ]
			then
				osflavor=$(grep -m 1 "Linux version" "${i}" | awk -F'(' '{print $4}' | awk -F')' '{print $1}')
			fi
			break
		fi
	done

	# --------------------------------------
	# Obtain kernel information
	# --------------------------------------
	
	kernelversion=$(uname -sr)

	# --------------------------------------
	# Determine OS and hardware architechture
	# --------------------------------------
	
	if [ `uname -i | grep -c "64"` -gt 0 ]
	then
		osbit="64"
	else
		osbit="32"
	fi

	if [ `grep -m 1 flags /proc/cpuinfo | grep -c " lm "` -gt 0 ]
	then
		cpubit="64"
	else
		cpubit="32"
	fi

	# --------------------------------------
	# Obtain CPU model and number of CPUs
	# --------------------------------------

	numcpus=$(grep -c processor /proc/cpuinfo)
	cpumodel=$(grep -m 1 "model name" /proc/cpuinfo | sed 's/  //g;s/: /#/g;s/(R)//g;s/  / /g' | awk -F'#' '{print $2}')
	cpucores=$(grep -m 1 "cpu cores" /proc/cpuinfo | sed 's/  //g;s/: /#/g;s/  / /g' | awk -F'#' '{print $2}')
	totalcores=$(echo "scale=0;${numcpus}*${cpucores}" | bc -l)

	# --------------------------------------
	# Obtain CPU utilization stats
	# --------------------------------------

	vmstatcpu=$(vmstat -n 1 1 | grep -v "wa" | awk '{print $13" "$14" "$15" "$16}')
	cpuus=$(echo ${vmstatcpu} | awk '{print $1}')
	cpusy=$(echo ${vmstatcpu} | awk '{print $2}')
	cpuid=$(echo ${vmstatcpu} | awk '{print $3}')
	cpuwa=$(echo ${vmstatcpu} | awk '{print $4}')

	# --------------------------------------
	# Calculate memory utilization
	# --------------------------------------

	memtotalkb=$(grep -m 1 "MemTotal" /proc/meminfo | awk '{print $2}')
	memtotalgb=$(echo "${memtotalkb}/1024/1024" | bc -l | awk '{printf("%5.0f\n",$0)}' | awk '{print $1}')
	memfreekb=$(grep -m 1 "MemFree" /proc/meminfo | awk '{print $2}')
	memfreemb=$(echo "${memfreekb}/1024" | bc -l | awk '{printf("%5.0f\n",$0)}'|awk '{print $1}')
	memutil=$(echo "100*(${memtotalkb}-${memfreekb})/${memtotalkb}" | bc -l | awk '{printf("%5.0f\n",$0)}' | awk '{print $1}')

	# --------------------------------------
	# Obtain the number of DIMMs and calculate
	# their approximate size
	# --------------------------------------
	
	numdimms=$(grep -c ram /proc/diskstats)
	dimmsize=$(echo "${memtotalgb}/${numdimms}" | bc -l)

	if [ $(echo "${dimmsize} < 1"|bc) -eq 1 ]
	then
		dimmsize2=$(echo "${dimmsize}*1024" | bc -l | awk '{printf("%5.0f\n",$0)}' | awk '{print $1}')
		dimmunit="Kb"
	else
		dimmsize2=$(echo ${dimmsize} | awk '{printf("%5.0f\n",$0)}' | awk '{print $1}')
		dimmunit="Gb"
	fi

	# --------------------------------------
	# Calculate memory and data paging
	# --------------------------------------

	swaptotalkb=$(free | grep -m 1 "Swap:" | awk '{print $2}')
	swaptotalgb=$(echo "${swaptotalkb}/1024/1024" | bc -l | awk '{printf("%5.0f\n",$0)}'|awk '{print $1}')
	swapfreekb=$(free | grep -m 1 "Swap:" | awk '{print $4}')
	swaputil=$(echo "100*(${swaptotalkb}-${swapfreekb})/${swaptotalkb}" | bc -l | awk '{printf("%5.0f\n",$0)}'|awk '{print $1}')

	vmstatout=$(vmstat -n 1 1 | egrep -v "memory|swpd")
	pswapin=$(echo "${vmstatout}" | awk '{print $7}')
	pswapout=$(echo "${vmstatout}" | awk '{print $8}')

	iobi=$(echo "${vmstatout}" | awk '{print $9}')
	iobo=$(echo "${vmstatout}" | awk '{print $10}')

	pswapinmax=$(grep -m 1 "pswpin" /proc/vmstat | awk '{print $2}')
	pswapoutmax=$(grep -m 1 "pswpout" /proc/vmstat | awk '{print $2}')

	# --------------------------------------
	# Claculate system average load based on
	# the available number of CPUs
	# --------------------------------------


	sysuptime=$(uptime | sed 's/^ //g;s/ up /#/g;s/, /#/g' | awk -F'#' '{print $2}')
	sysload1=$(awk '{print $1}' /proc/loadavg)
	sysload5=$(awk '{print $2}' /proc/loadavg)
	sysload15=$(awk '{print $3}' /proc/loadavg)

	sysload1true=$(echo "scale=2;${sysload1}/${totalcores}" | bc -l)
	sysload5true=$(echo "scale=2;${sysload5}/${totalcores}" | bc -l)
	sysload15true=$(echo "scale=2;${sysload15}/${totalcores}" | bc -l)

	if [ $(echo "${sysload15true} > 1"|bc) -eq 1 ]
	then
		cpuload15=$(echo "scale=0;(${sysload15true}-1)*100" | bc -l | awk '{printf("%5.0f\n",$0)}'|awk '{print $1}')
	else
		cpuload15=$(echo "scale=0;${sysload15true}*100" | bc -l | awk '{printf("%5.0f\n",$0)}'|awk '{print $1}')
	fi
	
	# --------------------------------------
	# Obtain versions of key Web-related apps
	# --------------------------------------


	mysqlver=$(mysql --version | grep Distrib | awk '{print $5}' | sed 's/,//g')
	perlver=$(perl --version | grep built | awk '{print $4}' | sed 's/v//g')
	#phpver=$(php --version | grep built | awk '{print $2}')

	# --------------------------------------
	# Calculate filesystem I/O performance
	# --------------------------------------

	blockfile=/tmp/blocksizefind.`perl -e 'print time, "\n"'`
	blocksize=$(echo "scale=0;`echo 'abc' > ${blockfile} ; du -h ${blockfile} | awk -F'.' '{print $1}'`*1024" | bc)
	rm /tmp/blocksizefind.* 2>/dev/null

	blockdevavg=""
	interval_count="0 1 2 3 4"
	interval_duration=1
	egrep [0-9] /proc/partitions | awk '{print $4}' | sed 's/[0-9]//g' | sort | uniq | while read blockdev
	do
		iostatline=$(iostat -xtd ${blockdev} 1 1 | grep "^${blockdev}")
		ioutilboot=$(echo ${iostatline} | awk '{print $14}')
		svctmboot=$(echo ${iostatline} | awk '{print $13}')

		for i in ${interval_count}
		do
			iostatline=$(iostat -xtd ${blockdev} ${interval_duration} 2 | grep "^${blockdev}" | tail -1)
			arrayutil[$i]=$(echo ${iostatline} | awk '{print $14}')
			arraysvct[$i]=$(echo ${iostatline} | awk '{print $13}')
		done

		ioutilavg=$(echo "scale=2; (${arrayutil[0]}+${arrayutil[1]}+${arrayutil[2]}+${arrayutil[3]}+${arrayutil[4]})/5" | bc -l)
		svctmavg=$(echo "scale=2; (${arraysvct[0]}+${arraysvct[1]}+${arraysvct[2]}+${arraysvct[3]}+${arraysvct[4]})/5" | bc -l)
		blockdevavg=$(${blockdevavg}; echo -n "${blockdev}: ${ioutilavg}%, ${svctmavg}ms")
		echo -n "${blockdev}: ${ioutilavg}%, ${svctmavg}ms; " >> ${blockfile}
	done
}

bottleneck() {
	problem=""
	if [ ${cpuwa} -gt 10 ]
	then
		problem=$(echo "${problem}, slow disk I/O")
	fi

	if [ ${cpuid} -lt 5 ]
	then
		problem=$(echo "${problem}, CPU overtaxed")
	fi

	if [ ${pswapin} -gt 100 ]
	then
		problem=$(echo "${problem}, insufficient RAM")
	elif [ ${pswapinmax} -gt 5000 ]
	then
		problem=$(echo "${problem}, occasional RAM shortages")
	fi

	problem=$(echo "${problem}" | sed 's/^, //g')
}

show() {
outfile=/tmp/diddle.`perl -e 'print time, "\n"'`
cat << EOF > "${outfile}"
Server:	${sysname}
OS    :	${osflavor}
Kernel:	${kernelversion}
Arch  :	${osbit}-bit OS running on ${cpubit}-bit hardware 
CPU(s):	${numcpus} x ${cpucores}-core ${cpumodel}
RAM   :	${memtotalgb}Gb (${memutil}% used), ${numdimms} x ${dimmsize2}${dimmunit} DIMMs
Swap  :	${swaptotalgb}Gb (${swaputil}% used), paging in/out: ${pswapin}/${pswapout}
Uptime:	${sysuptime}
Load  :	${sysload1true}, ${sysload5true}, ${sysload15true}
CPU % :	${totalcores} CPU cores at ${cpuload15}% combined utilization
Disk  :	util, svc time - `cat ${blockfile}`
Apps  :	MySQL ${mysqlver}, Perl ${perlver}
Issues:	${problem}
EOF
#rm /tmp/${blockfile} 2>/dev/null
cat ${outfile}
}

# --------------------------------------
# RUNTIME
# --------------------------------------

conf

if [ `ps -ef | grep -c "[d]iddle.ksh"` -le 2 ]
then
	if [ `ls -t /tmp | grep -c "diddle."` -gt 0 ]
	then
		timenow=$(perl -e 'print time, "\n"')
		timethen=$(ls /tmp | grep -m 1 diddle | awk -F'.' '{print $2}')
		timediff=$(echo "scale=0;${timenow}-${timethen}" | bc -l)
		if [ `echo "scale=0;${timenow}-${timethen}" | bc -l` -gt ${cache} ]
		then
			perf
			bottleneck
			rm /tmp/diddle* 2>/dev/null
			show
		else
			cat /tmp/`ls -t /tmp | grep -m 1 diddle`
		fi
	else
		perf
		bottleneck
		show
	fi
else
	if [ `ls -t /tmp | grep -c "diddle."` -gt 0 ]
	then
		cat /tmp/`ls -t /tmp | grep -m 1 "diddle."`
	fi
fi
